//  Copyright (C) 2001  MandrakeSoft S.A.
//
//    MandrakeSoft S.A.
//    43, rue d'Aboukir
//    75002 Paris - France
//    http://www.linux-mandrake.com/
//    http://www.mandrakesoft.com/
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2 of the License, or (at your option) any later version.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA







#include "bochs.h"
#include "assert.h"
#define LOG_THIS BX_MEM_THIS



#if BX_PROVIDE_CPU_MEMORY

  void
BX_MEM_C::write_physical(BX_CPU_C *cpu, Bit32u addr, unsigned len, void *data)
{
  Bit8u *data_ptr;
  Bit32u a20addr;


  a20addr = A20ADDR(addr);
  BX_INSTR_PHY_WRITE(a20addr, len);

#if BX_DEBUGGER
  // (mch) Check for physical write break points, TODO
  // (bbd) Each breakpoint should have an associated CPU#, TODO
  for (int i = 0; i < num_write_watchpoints; i++)
        if (write_watchpoint[i] == a20addr) {
              BX_CPU(0)->break_point = BREAK_POINT_WRITE;
              break;
        }
#endif


  if ( (a20addr + len) <= BX_MEM_THIS len ) {
    // all of data is within limits of physical memory
    if ( (a20addr & 0xfff80000) != 0x00080000 ) {
      if (len == 4) {
#if BX_NEED_DWORD_ALIGN
        if ((a20addr & 0x00000003) == 0) {
          // write 4byte data to aligned memory location
          Bit32u data32;

          data32 = * (Bit32u *) data;
#ifdef BX_BIG_ENDIAN
          data32 = (data32 << 24) | (data32 >> 24) |
            ((data32&0x00ff0000)>>8) | ((data32&0x0000ff00)<<8);
#endif
          * ((Bit32u *) (&vector[a20addr])) = data32;
          BX_DBG_DIRTY_PAGE(a20addr >> 12);
          BX_DYN_DIRTY_PAGE(a20addr >> 12);
          return;
          }
        else {
          Bit32u data32;

          data32 = * (Bit32u *) data;
          * ((Bit8u *) (&vector[a20addr]))         = data32; data32 >>= 8;
          BX_DBG_DIRTY_PAGE(a20addr >> 12);
          BX_DYN_DIRTY_PAGE(a20addr >> 12);
          * ((Bit8u *) (&vector[A20ADDR(addr+1)])) = data32; data32 >>= 8;
          * ((Bit8u *) (&vector[A20ADDR(addr+2)])) = data32; data32 >>= 8;
          * ((Bit8u *) (&vector[A20ADDR(addr+3)])) = data32;
          // worst case, last byte is in different page; possible extra dirty page
          BX_DBG_DIRTY_PAGE(A20ADDR(addr+3) >> 12);
          BX_DYN_DIRTY_PAGE(a20addr >> 12);
          return;
          }
#else /* BX_NEED_DWORD_ALIGN */
	Bit32u data32;

	data32 = * (Bit32u *) data;
#ifdef BX_BIG_ENDIAN
	data32 = (data32 << 24) | (data32 >> 24) |
	  ((data32&0x00ff0000)>>8) | ((data32&0x0000ff00)<<8);
#endif
	* ((Bit32u *) (&vector[a20addr])) = data32;
	BX_DBG_DIRTY_PAGE(a20addr >> 12);
	BX_DYN_DIRTY_PAGE(a20addr >> 12);
        if ((a20addr & 0x00000003) != 0) {
          // worst case, last byte is in different page; possible extra dirty page
          BX_DBG_DIRTY_PAGE(A20ADDR(addr+3) >> 12);
          BX_DYN_DIRTY_PAGE(a20addr >> 12);
	}
	return;
#endif /* BX_NEED_DWORD_ALIGN */
        }
      if (len == 2) {
#if BX_NEED_WORD_ALIGN
        if ((a20addr & 0x00000001) == 0) {
          // write 2-byte data to aligned memory location
          Bit16u data16;

          data16 = * (Bit16u *) data;
#ifdef BX_BIG_ENDIAN
          data16 = (data16 >> 8) | (data16 << 8);
#endif
          * ((Bit16u *) (&vector[a20addr])) = data16;
          BX_DBG_DIRTY_PAGE(a20addr >> 12);
          BX_DYN_DIRTY_PAGE(a20addr >> 12);
          return;
          }
        else {
          Bit16u data16;

          data16 = * (Bit16u *) data;
          * ((Bit8u *) (&vector[a20addr])) = (Bit8u) data16;
          BX_DBG_DIRTY_PAGE(a20addr >> 12);
          BX_DYN_DIRTY_PAGE(a20addr >> 12);
          * ((Bit8u *) (&vector[A20ADDR(a20addr+1)])) = (data16 >> 8);
          BX_DBG_DIRTY_PAGE(A20ADDR(a20addr+1) >> 12);
          BX_DYN_DIRTY_PAGE(a20addr >> 12);
          return;
          }
#else /* BX_NEED_WORD_ALIGN */
	Bit16u data16;

	data16 = * (Bit16u *) data;
#ifdef BX_BIG_ENDIAN
	data16 = (data16 >> 8) | (data16 << 8);
#endif
	* ((Bit16u *) (&vector[a20addr])) = data16;
	BX_DBG_DIRTY_PAGE(a20addr >> 12);
	BX_DYN_DIRTY_PAGE(a20addr >> 12);
        if ((a20addr & 0x00000001) != 0) {
          BX_DBG_DIRTY_PAGE(A20ADDR(a20addr+1) >> 12);
          BX_DYN_DIRTY_PAGE(a20addr >> 12);
	}
	return;
#endif /* BX_NEED_WORD_ALIGN */
        }
      if (len == 1) {
        Bit8u data8;

        data8 = * (Bit8u *) data;
        * ((Bit8u *) (&vector[a20addr])) = data8;
        BX_DBG_DIRTY_PAGE(a20addr >> 12);
        BX_DYN_DIRTY_PAGE(a20addr >> 12);
        return;
        }
      // len == 3 case can just fall thru to special cases handling
      }
#if BX_EMULATION_TOWNS
    }

#ifdef BX_LITTLE_ENDIAN
  data_ptr = (Bit8u *) data;
#else // BX_BIG_ENDIAN
  data_ptr = (Bit8u *) data + (len - 1);
#endif

write_one:
    switch(a20addr & 0xfffc0000) {
    case 0x000c0000:
      switch(a20addr & 0xf8000) {
      case 0xc0000:
	if(BX_MEM_THIS MAIN_MEM) goto mainmem;
	// FMR compatible GVRAM
	BX_VGA_MEM_WRITE(a20addr, *data_ptr);
	BX_DBG_DIRTY_PAGE(a20addr >> 12);
	BX_DYN_DIRTY_PAGE(a20addr >> 12);
	goto inc_one;
      case 0xc8000:
	if(BX_MEM_THIS MAIN_MEM) goto mainmem;
	if(a20addr>=0xcff80 && a20addr<=0xcffdf) {
	  // memory mapped I/O
	  BX_VGA_MEM_WRITE(a20addr, *data_ptr);
	  BX_DBG_DIRTY_PAGE(a20addr >> 12);
	  BX_DYN_DIRTY_PAGE(a20addr >> 12);
	  goto inc_one;
	}
	if(a20addr>=0xca000 && a20addr<=0xca7ff && BX_MEM_THIS ANKCG) {
	  // ANK CG ROM (8x8)
	  goto writerom;
	}
	if(a20addr>=0xcb000 && a20addr<=0xcbfff && BX_MEM_THIS ANKCG) {
	  // ANK CG ROM (8x16)
	  goto writerom;
	}
	if((a20addr>=0xc8000 && a20addr<=0xc8fff) ||
	   (a20addr>=0xca000 && a20addr<=0xcafff)) {
	  // FMR compatible TVRAM
	  BX_VGA_MEM_WRITE(a20addr, *data_ptr);
	  BX_DBG_DIRTY_PAGE(a20addr >> 12);
	  BX_DYN_DIRTY_PAGE(a20addr >> 12);
	  goto inc_one;
	}
	goto outrange;
      case 0xd0000:
	if(BX_MEM_THIS MAIN_MEM) goto mainmem;
	// DIC ROM
	goto writerom;
      case 0xd8000:
	if(BX_MEM_THIS MAIN_MEM) goto mainmem;
	// CMOS
	if(a20addr>=0xd8000 && a20addr<=0xd9fff) {
	  bx_devices.cmos->s.reg[a20addr-0xd8000]=*data_ptr;
	  BX_DBG_DIRTY_PAGE(a20addr >> 12);
	  BX_DYN_DIRTY_PAGE(a20addr >> 12);
	  goto inc_one;
	}
	goto outrange;
      case 0xe0000:
      case 0xe8000:
	if(BX_MEM_THIS MAIN_MEM) goto mainmem;
	goto outrange;
      case 0xf0000:
	goto mainmem;
      case 0xf8000:
	if(BX_MEM_THIS RAM) goto mainmem;
	// BOOT ROM
	goto writerom;
      }
    case 0x80000000:
    case 0x80040000:
    case 0x80100000:
    case 0x80140000:
      // VRAM
      BX_VGA_MEM_WRITE(a20addr, *data_ptr);
      BX_DBG_DIRTY_PAGE(a20addr >> 12);
      BX_DYN_DIRTY_PAGE(a20addr >> 12);
      goto inc_one;
    case 0x81000000:
      if(a20addr<0x81020000) {
	// sprite RAM
	BX_VGA_MEM_WRITE(a20addr, *data_ptr);
	BX_DBG_DIRTY_PAGE(a20addr >> 12);
	BX_DYN_DIRTY_PAGE(a20addr >> 12);
	goto inc_one;
      }
      goto outrange;
    case 0xc2000000:
    case 0xc2040000:
      // OS ROM
      goto writerom;
    case 0xc2080000:
    case 0xc20c0000:
      // DIC ROM
      goto writerom;
    case 0xc2100000:
      // FONT ROM
      goto writerom;
    case 0xc2140000:
      if(a20addr<0xc2142000) {
	// CMOS
	bx_devices.cmos->s.reg[a20addr-0xc2140000]=*data_ptr;
	BX_DBG_DIRTY_PAGE(a20addr >> 12);
	BX_DYN_DIRTY_PAGE(a20addr >> 12);
	goto inc_one;
      }
      goto outrange;
    case 0xc2200000:
      if(a20addr<0xc2201000) {
	// WAVE RAM
	bx_devices.fmsound->mem_write(a20addr,*data_ptr);
	BX_DBG_DIRTY_PAGE(a20addr >> 12);
	BX_DYN_DIRTY_PAGE(a20addr >> 12);
	goto inc_one;
      }
      goto outrange;
    case 0xfffc0000:
      // SYSTEM ROM
      goto writerom;
    default:
    if (a20addr < BX_MEM_THIS len) {
    mainmem:
      vector[a20addr] = *data_ptr;
      BX_DBG_DIRTY_PAGE(a20addr >> 12);
      BX_DYN_DIRTY_PAGE(a20addr >> 12);
inc_one:
      if (len == 1) return;
      len--;
      addr++;
      a20addr = A20ADDR(addr);
#ifdef BX_LITTLE_ENDIAN
      data_ptr++;
#else // BX_BIG_ENDIAN
      data_ptr--;
#endif
      goto write_one;
      }
    outrange:
    BX_INFO(("write 0x%02x to unmapped memory 0x%08x\n", *data_ptr, (unsigned) a20addr));
    BX_DBG_DIRTY_PAGE(a20addr >> 12);
    BX_DYN_DIRTY_PAGE(a20addr >> 12);
    goto inc_one;
    writerom:
    BX_INFO(("ROM lock 0x%08x <= 0x%02x\n", (unsigned) a20addr, *data_ptr));
    BX_DBG_DIRTY_PAGE(a20addr >> 12);
    BX_DYN_DIRTY_PAGE(a20addr >> 12);
    goto inc_one;
    }
#else

#ifdef BX_LITTLE_ENDIAN
  data_ptr = (Bit8u *) data;
#else // BX_BIG_ENDIAN
  data_ptr = (Bit8u *) data + (len - 1);
#endif

write_one:
    if ( (a20addr & 0xfff80000) != 0x00080000 ) {
      // addr *not* in range 00080000 .. 000FFFFF
      vector[a20addr] = *data_ptr;
      BX_DBG_DIRTY_PAGE(a20addr >> 12);
      BX_DYN_DIRTY_PAGE(a20addr >> 12);
inc_one:
      if (len == 1) return;
      len--;
      addr++;
      a20addr = A20ADDR(addr);
#ifdef BX_LITTLE_ENDIAN
      data_ptr++;
#else // BX_BIG_ENDIAN
      data_ptr--;
#endif
      goto write_one;
      }

    // addr in range 00080000 .. 000FFFFF

    if (a20addr <= 0x0009ffff) {
      // regular memory 80000 .. 9FFFF
      vector[a20addr] = *data_ptr;
      BX_DBG_DIRTY_PAGE(a20addr >> 12);
      BX_DYN_DIRTY_PAGE(a20addr >> 12);
      goto inc_one;
      }
    if (a20addr <= 0x000bffff) {
      // VGA memory A0000 .. BFFFF
      BX_VGA_MEM_WRITE(a20addr, *data_ptr);
      BX_DBG_DIRTY_PAGE(a20addr >> 12);
      BX_DYN_DIRTY_PAGE(a20addr >> 12);
      BX_DBG_UCMEM_REPORT(a20addr, 1, BX_WRITE, *data_ptr); // obsolete
      goto inc_one;
      }
    // adapter ROM     C0000 .. DFFFF
    // ROM BIOS memory E0000 .. FFFFF
    // (ignore write)
    //BX_INFO(("ROM lock %08x: len=%u\n",
    //  (unsigned) a20addr, (unsigned) len));
#if BX_PCI_SUPPORT == 0
#if BX_SHADOW_RAM
    // Write it since its in shadow RAM
    vector[a20addr] = *data_ptr;
    BX_DBG_DIRTY_PAGE(a20addr >> 12);
    BX_DYN_DIRTY_PAGE(a20addr >> 12);
#else
    // ignore write to ROM
#endif
#else
    // Write Based on 440fx Programming
    if (bx_options.i440FXSupport &&
        ((a20addr >= 0xC0000) && (a20addr <= 0xFFFFF))) {
      switch (bx_devices.pci->wr_memType(a20addr & 0xFC000)) {
        case 0x0:   // Writes to ShadowRAM
//        BX_INFO(("Writing to ShadowRAM %08x, len %u ! \n", (unsigned) a20addr, (unsigned) len));
          vector[a20addr] = *data_ptr;
          BX_DBG_DIRTY_PAGE(a20addr >> 12);
          BX_DYN_DIRTY_PAGE(a20addr >> 12);
          goto inc_one;

        case 0x1:   // Writes to ROM, Inhibit
//        bx_pci.s.i440fx.shadow[(a20addr - 0xc0000)] = *data_ptr;
//        BX_INFO(("Writing to ROM %08x, Data %02x ! \n", (unsigned) a20addr, *data_ptr));
          goto inc_one;
        default:
          BX_PANIC(("write_physical: default case\n"));
          goto inc_one;
        }
      }
#endif
    goto inc_one;
    }

  else {
    // some or all of data is outside limits of physical memory
    unsigned i;

#ifdef BX_LITTLE_ENDIAN
  data_ptr = (Bit8u *) data;
#else // BX_BIG_ENDIAN
  data_ptr = (Bit8u *) data + (len - 1);
#endif

#if BX_SUPPORT_APIC
    bx_generic_apic_c *local_apic = &cpu->local_apic;
    bx_generic_apic_c *ioapic = bx_devices.ioapic;
    if (local_apic->is_selected (a20addr, len)) {
      local_apic->write (a20addr, (Bit32u *)data, len);
      return;
    } else if (ioapic->is_selected (a20addr, len)) {
      ioapic->write (a20addr, (Bit32u *)data, len);
      return;
    }
    else 
#endif
    for (i = 0; i < len; i++) {
      if (a20addr < BX_MEM_THIS len) {
        vector[a20addr] = *data_ptr;
        BX_DBG_DIRTY_PAGE(a20addr >> 12);
        BX_DYN_DIRTY_PAGE(a20addr >> 12);
        }
      // otherwise ignore byte, since it overruns memory
      addr++;
      a20addr = A20ADDR(addr);
#ifdef BX_LITTLE_ENDIAN
      data_ptr++;
#else // BX_BIG_ENDIAN
      data_ptr--;
#endif
      }
    return;
    }
#endif
}


  void
BX_MEM_C::read_physical(BX_CPU_C *cpu, Bit32u addr, unsigned len, void *data)
{
  Bit8u *data_ptr;
  Bit32u a20addr;


  a20addr = A20ADDR(addr);
  BX_INSTR_PHY_READ(a20addr, len);

#if BX_DEBUGGER
  // (mch) Check for physical read break points, TODO
  // (bbd) Each breakpoint should have an associated CPU#, TODO
  for (int i = 0; i < num_read_watchpoints; i++)
        if (read_watchpoint[i] == a20addr) {
              BX_CPU(0)->break_point = BREAK_POINT_READ;
              break;
        }
#endif

  if ( (a20addr + len) <= BX_MEM_THIS len ) {
    // all of data is within limits of physical memory
    if ( (a20addr & 0xfff80000) != 0x00080000 ) {
      if (len == 4) {
#if BX_NEED_DWORD_ALIGN
        if ((a20addr & 0x00000003) == 0) {
#endif /* BX_NEED_DWORD_ALIGN */
          // read 4-byte data from aligned memory location
          Bit32u data32;

          data32 = * ((Bit32u *) (&vector[a20addr]));
#ifdef BX_BIG_ENDIAN
          data32 = (data32 << 24) | (data32 >> 24) |
                   ((data32&0x00ff0000)>>8) | ((data32&0x0000ff00)<<8);
#endif
          * (Bit32u *) data = data32;
          return;
#if BX_NEED_DWORD_ALIGN
          }
        else {
          Bit32u data32;

          data32  = * ((Bit8u *) (&vector[A20ADDR(addr+3)])); data32 <<= 8;
          data32 |= * ((Bit8u *) (&vector[A20ADDR(addr+2)])); data32 <<= 8;
          data32 |= * ((Bit8u *) (&vector[A20ADDR(addr+1)])); data32 <<= 8;
          data32 |= * ((Bit8u *) (&vector[a20addr]));

          * (Bit32u *) data = data32;
          return;
          }
#endif /* BX_NEED_DWORD_ALIGN */
        }
      if (len == 2) {
#if BX_NEED_WORD_ALIGN
        if ((a20addr & 0x00000001) == 0) {
#endif /* BX_NEED_WORD_ALIGN */
          // read 2-byte data from aligned memory location
          Bit16u data16;

          data16 = * ((Bit16u *) (&vector[a20addr]));
#ifdef BX_BIG_ENDIAN
          data16 = (data16 >> 8) | (data16 << 8);
#endif

          * (Bit16u *) data =  data16;
          return;
#if BX_NEED_WORD_ALIGN
          }
        else {
          Bit16u data16;

          data16  = * ((Bit8u *) (&vector[A20ADDR(addr+1)])); data16 <<= 8;
          data16 |= * ((Bit8u *) (&vector[a20addr]));

          * (Bit16u *) data = data16;
          return;
          }
#endif /* BX_NEED_WORD_ALIGN */
        }
      if (len == 1) {
        Bit8u data8;

        data8 = * ((Bit8u *) (&vector[a20addr]));
        * (Bit8u *) data = data8;
        return;
        }
      // len == 3 case can just fall thru to special cases handling
      }
#if BX_EMULATION_TOWNS
    }


#ifdef BX_LITTLE_ENDIAN
    data_ptr = (Bit8u *) data;
#else // BX_BIG_ENDIAN
    data_ptr = (Bit8u *) data + (len - 1);
#endif



read_one:
    switch(a20addr & 0xfffc0000) {
    case 0x000c0000:
      switch(a20addr & 0xf8000) {
      case 0xc0000:
	if(BX_MEM_THIS MAIN_MEM) goto mainmem;
	// FMR compatible GVRAM
	*data_ptr = BX_VGA_MEM_READ(a20addr);
	goto inc_one;
      case 0xc8000:
	if(BX_MEM_THIS MAIN_MEM) goto mainmem;
	if(a20addr>=0xcff80 && a20addr<=0xcffdf) {
	  // memory mapped I/O
	  *data_ptr = BX_VGA_MEM_READ(a20addr);
	  goto inc_one;
	}
	if(a20addr>=0xca000 && a20addr<=0xca7ff && BX_MEM_THIS ANKCG) {
	  // ANK CG ROM (8x8)
	  *data_ptr = BX_MEM_THIS fntrom[(a20addr&0x7ff)+0x3d000];
	  goto inc_one;
	}
	if(a20addr>=0xcb000 && a20addr<=0xcbfff && BX_MEM_THIS ANKCG) {
	  // ANK CG ROM (8x16)
	  *data_ptr = BX_MEM_THIS fntrom[(a20addr&0xfff)+0x3d800];
	  goto inc_one;
	}
	if((a20addr>=0xc8000 && a20addr<=0xc8fff) ||
	   (a20addr>=0xca000 && a20addr<=0xcafff)) {
	  // FMR compatible TVRAM
	  *data_ptr = BX_VGA_MEM_READ(a20addr);
	  goto inc_one;
	}
	goto outrange;
      case 0xd0000:
	if(BX_MEM_THIS MAIN_MEM) goto mainmem;
	// DIC ROM
	goto outrange;
      case 0xd8000:
	if(BX_MEM_THIS MAIN_MEM) goto mainmem;
	// CMOS
	if(a20addr>=0xd8000 && a20addr<=0xd9fff) {
	  *data_ptr=bx_devices.cmos->s.reg[a20addr-0xd8000];
	  goto inc_one;
	}
	goto outrange;
      case 0xe0000:
      case 0xe8000:
	if(BX_MEM_THIS MAIN_MEM) goto mainmem;
	goto outrange;
      case 0xf0000:
	goto mainmem;
      case 0xf8000:
	if(BX_MEM_THIS RAM) goto mainmem;
	// BOOT ROM
	*data_ptr = sysrom[a20addr-(0xf8000-0x38000)];
	goto inc_one;
      }
    case 0x80000000:
    case 0x80040000:
    case 0x80100000:
    case 0x80140000:
      // VRAM
      *data_ptr = BX_VGA_MEM_READ(a20addr);
      goto inc_one;
    case 0x81000000:
      if(a20addr<0x81020000) {
	// sprite RAM
	*data_ptr = BX_VGA_MEM_READ(a20addr);
	goto inc_one;
      }
      goto outrange;
    case 0xc2000000:
    case 0xc2040000:
      // OS ROM
      *data_ptr = dosrom[a20addr-0xc2000000];
      goto inc_one;
    case 0xc2080000:
    case 0xc20c0000:
      // DIC ROM
      *data_ptr = dicrom[a20addr-0xc2080000];
      goto inc_one;
    case 0xc2100000:
      // FONT ROM
      *data_ptr = fntrom[a20addr-0xc2100000];
      goto inc_one;
    case 0xc2140000:
      if(a20addr<0xc2142000) {
	// CMOS
	*data_ptr=bx_devices.cmos->s.reg[a20addr-0xc2140000];
	goto inc_one;
      }
      goto outrange;
    case 0xc2200000:
      if(a20addr<0xc2201000) {
	// WAVE RAM
	*data_ptr=bx_devices.fmsound->mem_read(a20addr);
	goto inc_one;
	// WAVE RAM
      }
      goto outrange;
    case 0xfffc0000:
      // SYSTEM ROM
      *data_ptr = sysrom[a20addr-0xfffc0000];
      goto inc_one;
    default:
    if (a20addr < BX_MEM_THIS len) {
    mainmem:
      *data_ptr = vector[a20addr];
inc_one:
      if (len == 1) return;
      len--;
      addr++;
      a20addr = A20ADDR(addr);
#ifdef BX_LITTLE_ENDIAN
      data_ptr++;
#else // BX_BIG_ENDIAN
      data_ptr--;
#endif
      goto read_one;
      }
    outrange:
    *data_ptr = 0xff;
    BX_INFO(("read from unmapped memory 0x%08x\n", (unsigned) a20addr));
    goto inc_one;
    }
#else


#ifdef BX_LITTLE_ENDIAN
    data_ptr = (Bit8u *) data;
#else // BX_BIG_ENDIAN
    data_ptr = (Bit8u *) data + (len - 1);
#endif



read_one:
    if ( (a20addr & 0xfff80000) != 0x00080000 ) {
      // addr *not* in range 00080000 .. 000FFFFF
      *data_ptr = vector[a20addr];
inc_one:
      if (len == 1) return;
      len--;
      addr++;
      a20addr = A20ADDR(addr);
#ifdef BX_LITTLE_ENDIAN
      data_ptr++;
#else // BX_BIG_ENDIAN
      data_ptr--;
#endif
      goto read_one;
      }

    // addr in range 00080000 .. 000FFFFF
#if BX_PCI_SUPPORT == 0
    if ((a20addr <= 0x0009ffff) || (a20addr >= 0x000c0000) ) {
      // regular memory 80000 .. 9FFFF, C0000 .. F0000
      *data_ptr = vector[a20addr];
      goto inc_one;
      }
    // VGA memory A0000 .. BFFFF
    *data_ptr = BX_VGA_MEM_READ(a20addr);
    BX_DBG_UCMEM_REPORT(a20addr, 1, BX_READ, *data_ptr); // obsolete
    goto inc_one;
#else   // #if BX_PCI_SUPPORT == 0
    if (a20addr <= 0x0009ffff) {
      *data_ptr = vector[a20addr];
      goto inc_one;
      }
    if (a20addr <= 0x000BFFFF) {
      // VGA memory A0000 .. BFFFF
      *data_ptr = BX_VGA_MEM_READ(a20addr);
      BX_DBG_UCMEM_REPORT(a20addr, 1, BX_READ, *data_ptr);
      goto inc_one;
      }

    // a20addr in C0000 .. FFFFF
    if (!bx_options.i440FXSupport) {
      *data_ptr = vector[a20addr];
      goto inc_one;
      }
    else {
      switch (bx_devices.pci->rd_memType(a20addr & 0xFC000)) {
        case 0x0:   // Read from ShadowRAM
          *data_ptr = vector[a20addr];
          BX_INFO(("Reading from ShadowRAM %08x, Data %02x \n", (unsigned) a20addr, *data_ptr));
          goto inc_one;

        case 0x1:   // Read from ROM
          *data_ptr = bx_pci.s.i440fx.shadow[(a20addr - 0xc0000)];
          //BX_INFO(("Reading from ROM %08x, Data %02x  \n", (unsigned) a20addr, *data_ptr));
          goto inc_one;
        default:
          BX_PANIC(("::read_physical: default case\n"));
        }
      }
    goto inc_one;
#endif  // #if BX_PCI_SUPPORT == 0
    }
  else {
    // some or all of data is outside limits of physical memory
    unsigned i;

#ifdef BX_LITTLE_ENDIAN
    data_ptr = (Bit8u *) data;
#else // BX_BIG_ENDIAN
    data_ptr = (Bit8u *) data + (len - 1);
#endif

#if BX_SUPPORT_APIC
    bx_generic_apic_c *local_apic = &cpu->local_apic;
    bx_generic_apic_c *ioapic = bx_devices.ioapic;
    if (local_apic->is_selected (addr, len)) {
      local_apic->read (addr, data, len);
      return;
    } else if (ioapic->is_selected (addr, len)) {
      ioapic->read (addr, data, len);
      return;
    }
#endif
    for (i = 0; i < len; i++) {
#if BX_PCI_SUPPORT == 0
      if (a20addr < BX_MEM_THIS len)
        *data_ptr = vector[a20addr];
      else
        *data_ptr = 0xff;
#else   // BX_PCI_SUPPORT == 0
      if (a20addr < BX_MEM_THIS len) {
        if ((a20addr >= 0x000C0000) && (a20addr <= 0x000FFFFF)) {
          if (!bx_options.i440FXSupport)
            *data_ptr = vector[a20addr];
          else {
            switch (bx_devices.pci->rd_memType(a20addr & 0xFC000)) {
              case 0x0:   // Read from ROM
                *data_ptr = vector[a20addr];
                //BX_INFO(("Reading from ROM %08x, Data %02x \n", (unsigned) a20addr, *data_ptr));
                break;

              case 0x1:   // Read from Shadow RAM
                *data_ptr = bx_pci.s.i440fx.shadow[(a20addr - 0xc0000)];
                BX_INFO(("Reading from ShadowRAM %08x, Data %02x  \n", (unsigned) a20addr, *data_ptr));
                break;
              default:
                BX_PANIC(("read_physical: default case\n"));
              } // Switch
            }
          }
        else {
          *data_ptr = vector[a20addr];
          BX_INFO(("Reading from Norm %08x, Data %02x  \n", (unsigned) a20addr, *data_ptr));
          }
        }
      else 
        *data_ptr = 0xff;
#endif  // BX_PCI_SUPPORT == 0
      addr++;
      a20addr = A20ADDR(addr);
#ifdef BX_LITTLE_ENDIAN
      data_ptr++;
#else // BX_BIG_ENDIAN
      data_ptr--;
#endif
      }
    return;
    }
#endif
}


#if BX_EMULATION_TOWNS
Bit8u*
BX_MEM_C::get_fetch_ptr(Bit32u phy_addr)
{
  //BX_INFO(("%08x\n", (unsigned) phy_addr));
  switch(phy_addr & 0xfffc0000) {
  case 0x000c0000:
    switch(phy_addr & 0xf8000) {
    case 0xc0000:
      if(BX_MEM_THIS MAIN_MEM) goto mainmem;
      // FMR compatible GVRAM
      goto outrange;
    case 0xc8000:
      if(BX_MEM_THIS MAIN_MEM) goto mainmem;
      // FMR compatible TVRAM
      goto outrange;
    case 0xd0000:
      if(BX_MEM_THIS MAIN_MEM) goto mainmem;
      // DIC ROM
      goto outrange;
    case 0xd8000:
      if(BX_MEM_THIS MAIN_MEM) goto mainmem;
      // CMOS
      goto outrange;
    case 0xe0000:
    case 0xe8000:
      if(BX_MEM_THIS MAIN_MEM) goto mainmem;
      goto outrange;
    case 0xf0000:
      goto mainmem;
    case 0xf8000:
      if(BX_MEM_THIS RAM) goto mainmem;
      // BOOT ROM
      return(&sysromx[phy_addr-(0xf8000-0x38000)]);
    }
  case 0x80000000:
  case 0x80040000:
  case 0x80100000:
  case 0x80140000:
    // VRAM
    goto outrange;
  case 0x81000000:
    if(phy_addr<0x81020000) {
      // sprite RAM
    }
    goto outrange;
  case 0xc2000000:
  case 0xc2040000:
    // OS ROM
    return(&dosrom[phy_addr-0xc2000000]);
  case 0xc2080000:
  case 0xc20c0000:
    // DIC ROM
    return(&dicrom[phy_addr-0xc2080000]);
  case 0xc2100000:
    // FONT ROM
    return(&fntrom[phy_addr-0xc2100000]);
  case 0xc2140000:
    if(phy_addr<0xc2142000) {
      // CMOS
    }
    goto outrange;
  case 0xc2200000:
    if(phy_addr<0xc2201000) {
      // WAVE RAM
    }
    goto outrange;
  case 0xfffc0000:
    // SYSTEM ROM
    return(&sysromx[phy_addr-0xfffc0000]);
  default:
    if (phy_addr < BX_MEM_THIS len) {
    mainmem:
      return(&vector[phy_addr]);
    }
  }
outrange:
  BX_PANIC(("prefetch: running in bogus memory\n"));
  return(0); // dummy
}


  void
BX_MEM_C::block_copy_physical(BX_CPU_C *cpu, 
			      laddrinfo dest,
			      laddrinfo src,
			      Bit32u len)
{
  Bit32u clen,endp,i;
  Bit8u *destptr,*srcptr;
  Bit8u tmp;

  BX_INFO(("block_copy_physical: len=0x%.8x.\n",len));
  if(src.vmem) {
    BX_INFO(("block_copy_physical: src=(0x%.8x).\n",src.a.l));
  } else {
    BX_INFO(("block_copy_physical: src=[%p]\n",src.a.p));
  }
  if(dest.vmem) {
    BX_INFO(("block_copy_physical: dest=(0x%.8x).\n",dest.a.l));
  } else {
    BX_INFO(("block_copy_physical: dest=[%p]\n",dest.a.p));
  }

  while(len > 0) {
    clen = len;

    if(src.vmem) {
      if(src.a.l < 0x000c0000) {
	srcptr = &BX_MEM_THIS vector[src.a.l];
	endp = 0x000bffff;
      } else if(src.a.l < 0x000f0000) {
	srcptr = BX_MEM_THIS MAIN_MEM ? &BX_MEM_THIS vector[src.a.l] : NULL;
	endp = 0x000effff;
      } else if(src.a.l < 0x000f8000) {
	srcptr = &BX_MEM_THIS vector[src.a.l];
	endp = 0x000f7fff;
      } else if(src.a.l < 0x00100000) {
	srcptr = BX_MEM_THIS RAM ? &BX_MEM_THIS vector[src.a.l] : NULL;
	endp = 0x000fffff;
      } else if(src.a.l < BX_MEM_THIS len) {
	srcptr = &BX_MEM_THIS vector[src.a.l];
	endp = BX_MEM_THIS len - 1;
      } else {
	srcptr = NULL;
	endp = 0xffffffff;
      }
      if(endp - src.a.l + 1 < clen) {
	clen = endp - src.a.l + 1;
      }
    } else {
      srcptr = (Bit8u*) src.a.p;
    }

    if(dest.vmem) {
      if(dest.a.l < 0x000c0000) {
	destptr = &BX_MEM_THIS vector[dest.a.l];
	endp = 0x000bffff;
      } else if(dest.a.l < 0x000f0000) {
	destptr = BX_MEM_THIS MAIN_MEM ? &BX_MEM_THIS vector[dest.a.l] : NULL;
	endp = 0x000effff;
      } else if(dest.a.l < 0x000f8000) {
	destptr = &BX_MEM_THIS vector[dest.a.l];
	endp = 0x000f7fff;
      } else if(dest.a.l < 0x00100000) {
	destptr = BX_MEM_THIS RAM ? &BX_MEM_THIS vector[dest.a.l] : NULL;
	endp = 0x000fffff;
      } else if(dest.a.l < BX_MEM_THIS len) {
	destptr = &BX_MEM_THIS vector[dest.a.l];
	endp = BX_MEM_THIS len - 1;
      } else {
	destptr = NULL;
	endp = 0xffffffff;
      }
      if(endp - dest.a.l + 1 < clen) {
	clen = endp - dest.a.l + 1;
      }
    } else {
      destptr = (Bit8u*) dest.a.p;
    }

    if(srcptr == NULL) {
      if(destptr == NULL) {
	for(i=0;i<clen;i++) {
	  read_physical(cpu,src.a.l+i,1,&tmp);
	  write_physical(cpu,dest.a.l+i,1,&tmp);
	}
      } else {
	for(i=0;i<clen;i++) {
	  read_physical(cpu,src.a.l+i,1,destptr+i);
	}
      }
    } else {
      if(destptr == NULL) {
	for(i=0;i<clen;i++) {
	  write_physical(cpu,dest.a.l+i,1,srcptr+i);
	}
      } else {
	memcpy(destptr,srcptr,clen);
      }
    }

    len -= clen;
    if(src.vmem) {
      src.a.l += clen;
    } else {
      src.a.p = ((Bit8u*)(src.a.p)) + clen;
    }
    if(dest.vmem) {
      dest.a.l += clen;
    } else {
      dest.a.p = ((Bit8u*)(dest.a.p)) + clen;
    }
  }
}
#endif

#if BX_SEGMAP
#undef BX_MEM_SMF
#undef BX_MEM_THIS
#undef LOG_THIS

Bit8u *bx_segment_addrmap_t::genmap_phys(BX_MEM_C *mem,
					 Bit32u physical,
					 Bit32u *max,
					 Bit32u *min,
					 Boolean write)
{
  //Bit8u *data_ptr;
  Bit32u a20addr;
  Bit32u amin,amax;
  Bit8u *p;

  a20addr = A20ADDR(physical);
  *min = A20ADDR(*min);
  *max = A20ADDR(*max);

#if BX_DEBUGGER
  // (mch) Check for physical read break points, TODO
  // (bbd) Each breakpoint should have an associated CPU#, TODO
  if(write) {
    for (int i = 0; i < num_write_watchpoints; i++)
      if (write_watchpoint[i] == a20addr) return NULL;
  } else {
    for (int i = 0; i < num_read_watchpoints; i++)
      if (read_watchpoint[i] == a20addr) return NULL;
  }
#endif

#if BX_EMULATION_TOWNS
  if(a20addr < 0x000c0000) {
    amin = 0x00000000;
    amax = 0x000bffff;
    goto mainram;
  } else if(a20addr < 0x000f0000) {
    return NULL;	// flush on bank change is not yet.
    if(!mem->MAIN_MEM) return NULL;
    amin = 0x000c0000;
    amax = 0x000effff;
    goto mainram;
  } else if(a20addr < 0x000f8000) {
    amin = 0x000f0000;
    amax = 0x000f7fff;
    goto mainram;
  } else if(a20addr < 0x00100000) {
    return NULL;	// flush on bank change is not yet.
    amin = 0x000f8000;
    amax = 0x000fffff;
    if(mem->MAIN_MEM) goto mainram;
    p = &mem->sysrom[a20addr-(0xf8000-0x38000)];
    goto othermem;
  } else if(a20addr < mem->len) {
    amin = 0x00100000;
    amax = mem->len-1;
    goto mainram;
  } else if(a20addr < 0xfffc0000) {
    return NULL;
  } else {
    amin = 0xfffc0000;
    amax = 0xffffffff;
    p = &mem->sysrom[a20addr-0xfffc0000];
    goto othermem;
  }
#else
  if(a20addr < 0x000a0000) {
    amin = 0x00000000;
    amax = 0x0009ffff;
    goto mainram;
  } else if(a20addr < 0x00100000) {
    return NULL;
  } else if(a20addr < mem->len) {
    amin = 0x00100000;
    amax = mem->len-1;
    goto mainram;
  } else {
    return NULL;
  }
#endif
mainram:
  p = &mem->vector[a20addr];
othermem:
  if(*min<amin) *min=amin;
  if(*max>amax) *max=amax;
  if(*min>a20addr) return NULL;
  if(a20addr>*max) return NULL;

  *min=*min-a20addr+physical;
  *max=*max-a20addr+physical;
  assert(*min<=physical);
  assert(physical<=*max);

  return p;
}
#endif

#endif // #if BX_PROVIDE_CPU_MEMORY
